// vi: set ts=4 sw=4 :
// vim: set tw=75 :

// meta_api.cpp - minimal implementation of metamod's plugin interface

// This is intended to illustrate the (more or less) bare minimum code
// required for a valid metamod plugin, and is targeted at those who want
// to port existing HL/SDK DLL code to run as a metamod plugin.

/*
 * Copyright (c) 2001-2003 Will Day <willday@hpgx.net>
 *
 *    This file is part of Metamod.
 *
 *    Metamod is free software; you can redistribute it and/or modify it
 *    under the terms of the GNU General Public License as published by the
 *    Free Software Foundation; either version 2 of the License, or (at
 *    your option) any later version.
 *
 *    Metamod is distributed in the hope that it will be useful, but
 *    WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with Metamod; if not, write to the Free Software Foundation,
 *    Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *    In addition, as a special exception, the author gives permission to
 *    link the code of this program with the Half-Life Game Engine ("HL
 *    Engine") and Modified Game Libraries ("MODs") developed by Valve,
 *    L.L.C ("Valve").  You must obey the GNU General Public License in all
 *    respects for all of the code used other than the HL Engine and MODs
 *    from Valve.  If you modify this file, you may extend this exception
 *    to your version of the file, but you are not obligated to do so.  If
 *    you do not wish to do so, delete this exception statement from your
 *    version.
 *
 */

#include <extdll.h>			// always

#include <meta_api.h>		// of course

#ifdef _WIN32
	#include "tlhelp32.h"
#endif

#ifdef LINUX
	#include <sys/mman.h>
	#ifndef DWORD
		typedef unsigned long DWORD;
	#endif
	#ifndef WORD
		typedef unsigned short WORD;
	#endif
	#ifndef PAGESIZE
		#define PAGESIZE 4096
	#endif
#endif

// Must provide at least one of these..
static META_FUNCTIONS gMetaFunctionTable = {
	NULL,				// pfnGetEntityAPI				HL SDK; called before game DLL
	NULL,				// pfnGetEntityAPI_Post			META; called after game DLL
	GetEntityAPI2,			// pfnGetEntityAPI2				HL SDK2; called before game DLL
	GetEntityAPI2_Post,			// pfnGetEntityAPI2_Post		META; called after game DLL
	NULL,			// pfnGetNewDLLFunctions		HL SDK2; called before game DLL
	NULL,			// pfnGetNewDLLFunctions_Post	META; called after game DLL
	NULL,			// pfnGetEngineFunctions	META; called before HL engine
	NULL,			// pfnGetEngineFunctions_Post	META; called after HL engine
};

#define MODULE_VERSION		"1.4"
#define MODULE_VERSION_F	1.4
#define MODULE_LOADABLE		PT_STARTUP
#define MODULE_UNLOADABLE	PT_NEVER

#define COMMAND_CONTINUE	0
#define COMMAND_HANDLED		1

// Description of plugin
plugin_info_t Plugin_info = {
	META_INTERFACE_VERSION,	// ifvers
	"AntiDlFile",	// name
	MODULE_VERSION,	// version
	"2010/10/18",	// date
	"Rulzy",	// author
	"http://www.dt-club.net/",	// url
	"ANTI-DLFILE",	// logtag, all caps please
	MODULE_LOADABLE,	// (when) loadable
	MODULE_UNLOADABLE,	// (when) unloadable
};

cvar_t MODULE_CVAR_VERSION = {
	"antidlfile_version",
	MODULE_VERSION,
	FCVAR_SERVER,
	MODULE_VERSION_F,
	NULL
};

cvar_t MODULE_CVAR_ENABLE = {
	"antidlfile_enable",
	"1",
	0,
	1.0,
	NULL
};

// Global vars from metamod:
meta_globals_t *gpMetaGlobals;		// metamod globals
gamedll_funcs_t *gpGamedllFuncs;	// gameDLL function tables
mutil_funcs_t *gpMetaUtilFuncs;		// metamod utility functions

struct CPlayer
{
	char Name[32];
	char Address[28];
	BOOL InGame;
} gPlayer[33];


char *FindMemory(char *Start, char *End, const char *Buffer, int BufferSize)
{
	for( char *p = Start; p <= End-BufferSize-1; p++ )
		if(!memcmp(p, Buffer, BufferSize))
			return p;
	return NULL;
}

struct CModuleInfo{
	void *BaseAddr;
	DWORD Size;
};

int HookClientEngineCommand(edict_t *pEntity, char *pCommandLine)
{
	if(!int(CVAR_GET_FLOAT(MODULE_CVAR_ENABLE.name)))
		return COMMAND_CONTINUE;
	char *p = pCommandLine;
	if(!p || !*p) return COMMAND_CONTINUE;
	while(*p<=' ' && *p) p++;
	if(!*p) return COMMAND_CONTINUE;
	if(!strncasecmp(p, "dlfile", 6) && *(p+6)<=' ')
		p+=6;
	else if ( !strncasecmp(p, "\"dlfile\"", 8))
		p+=8;
	else
		return COMMAND_CONTINUE;

	while(*p<=' ' && *p) p++;
	if(!*p) return COMMAND_CONTINUE;
	if( !strncmp(p, "!MD5", 4) || !strncmp(p, "\"!MD5", 5) ) return COMMAND_CONTINUE;
	int Index = ENTINDEX(pEntity);
	if( Index < 1 || Index > gpGlobals->maxClients )
		return COMMAND_CONTINUE;
	if( gPlayer[Index].InGame || !strncasecmp(p, "pak0.pak", 8) || !strncasecmp(p, "\"pak0.pak", 9) || !strncasecmp(p, "halflife.wad", 12) || !strncasecmp(p, "\"halflife.wad", 13) ) // pak0.pak and halflife.wad is too big, may cause server down!
	{
		LOG_MESSAGE(PLID, "BLOCKED: \"%s<%d><%s><>\" (address \"%s\") is using \"cmd %s\" command.\n",
			gPlayer[Index].Name, GETPLAYERUSERID(pEntity), GETPLAYERAUTHID(pEntity), gPlayer[Index].Address, pCommandLine);
		return COMMAND_HANDLED;
	}
	return COMMAND_CONTINUE;
}

BOOL C_ClientConnect(edict_t *pEntity, const char *pszName, const char *pszAddress, char szRejectReason[128])
{
	int Index = ENTINDEX(pEntity);
	if( Index>=1 && Index<=gpGlobals->maxClients )
	{
		strncpy(gPlayer[Index].Name, pszName, 31);
		gPlayer[Index].Name[31] = 0;
		strncpy(gPlayer[Index].Address, pszAddress, 27);
		gPlayer[Index].Address[27] = 0;
		gPlayer[Index].InGame = FALSE;
	}
	RETURN_META_VALUE(MRES_IGNORED, FALSE);
}

void C_ClientDisconnect(edict_t *pEntity)
{
	int Index = ENTINDEX(pEntity);
	if( Index>=1 && Index<=gpGlobals->maxClients )
		gPlayer[Index].InGame = FALSE;
	RETURN_META(MRES_IGNORED);
}

void C_ClientPutInServer_Post(edict_t *pEntity)
{
	int Index = ENTINDEX(pEntity);
	if( Index>=1 && Index<=gpGlobals->maxClients )
		gPlayer[Index].InGame = TRUE;
	RETURN_META(MRES_IGNORED);
}

void C_ClientUserInfoChanged_Post(edict_t *pEntity, char *infobuffer)
{
	int Index = ENTINDEX(pEntity);
	if( Index>=1 && Index<=gpGlobals->maxClients )
	{
		int BackSlashCount = 0;
		for( char *p = infobuffer; *p != '\0'; p++ )
		{
			if( *p == '\\')
				BackSlashCount++;
			if(BackSlashCount%2 && !strncasecmp(p, "\\name\\", 6))
			{
				p+=6;
				int count = 0;
				while(*p && *p!='\\' && count<31)
				{
					gPlayer[Index].Name[count] = *p;
					count++;
					p++;
				}
				gPlayer[Index].Name[count] = '\0';
				break;
			}
		}
	}
	RETURN_META(MRES_IGNORED);
}

void C_ServerDeactivate()
{
	for(int i=1; i<=gpGlobals->maxClients; i++)
		gPlayer[i].InGame = FALSE;
	RETURN_META(MRES_IGNORED);
}

DLL_FUNCTIONS gFunctionTable;
C_DLLEXPORT	int	GetEntityAPI2(DLL_FUNCTIONS *pFunctionTable, int *interfaceVersion)
{
	memset(&gFunctionTable, 0, sizeof(DLL_FUNCTIONS));

	gFunctionTable.pfnClientConnect = C_ClientConnect;
	gFunctionTable.pfnClientDisconnect = C_ClientDisconnect;
	gFunctionTable.pfnServerDeactivate = C_ServerDeactivate;

	memcpy(pFunctionTable, &gFunctionTable, sizeof(DLL_FUNCTIONS));

	return TRUE;
}

DLL_FUNCTIONS gFunctionTable_Post;
C_DLLEXPORT	int	GetEntityAPI2_Post(DLL_FUNCTIONS *pFunctionTable, int *interfaceVersion)
{
	memset(&gFunctionTable_Post, 0, sizeof(DLL_FUNCTIONS));

	gFunctionTable_Post.pfnClientPutInServer = C_ClientPutInServer_Post;
	gFunctionTable_Post.pfnClientUserInfoChanged = C_ClientUserInfoChanged_Post;

	memcpy(pFunctionTable, &gFunctionTable_Post, sizeof(DLL_FUNCTIONS));

	return TRUE;
}

C_DLLEXPORT int Meta_Query(char *ifvers, plugin_info_t **pPlugInfo,	mutil_funcs_t *pMetaUtilFuncs) 
{
	*pPlugInfo = &Plugin_info;
	gpMetaUtilFuncs = pMetaUtilFuncs;

	int	mmajor = 0, mminor = 0,	pmajor = 0, pminor = 0;

	sscanf(ifvers, "%d:%d",	&mmajor, &mminor);
	sscanf(Plugin_info.ifvers, "%d:%d",	&pmajor, &pminor);

	if (strcmp(ifvers, Plugin_info.ifvers))
	{
		if (pmajor > mmajor)
		{
			LOG_ERROR(PLID, "metamod version is too old for this plugin; update metamod");
			return FALSE;
		} else if (pmajor < mmajor) {
			LOG_ERROR(PLID, "metamod version is incompatible with this plugin; please find a newer version of this plugin");
			return FALSE;
		} else if (pmajor == mmajor) {
			if (pminor > mminor) 
			{
				LOG_ERROR(PLID, "metamod version is too old for this plugin; update metamod");
				return FALSE;
			} else if (pminor < mminor) {
				LOG_MESSAGE(PLID, "warning: meta-interface version mismatch (plugin: %s, metamod: %s)", Plugin_info.ifvers, ifvers);
			}
		}
	}
	return TRUE;
}

BYTE PatchCodeSpace[0x50];

const char *Flag = "\x83\xF8\x7F\x7E\x04\xC6\x46\x7F\x00";
// 83F8 7F         cmp     eax, 7F
// 7E 04           jle     +4
// C646 7F 00      mov     byte ptr [esi+7F], 0

#ifdef _WIN32
bool GetModuleInfo(const char *ModuleName, CModuleInfo *pMI )
{
	HANDLE hSnapshot = CreateToolhelp32Snapshot(TH32CS_SNAPMODULE, 0);
	if(hSnapshot == HANDLE(-1)) return false;
	MODULEENTRY32 ME;
	memset(&ME, 0, sizeof(MODULEENTRY32));
	ME.dwSize = sizeof(MODULEENTRY32);
	if(Module32First(hSnapshot, &ME))
	{
		if(!ModuleName || !*ModuleName)
		{
			pMI->BaseAddr = ME.modBaseAddr;
			pMI->Size = ME.modBaseSize;
			CloseHandle(hSnapshot);
			return true;
		}
		do
		{
			if(!strcmp(ME.szModule, ModuleName))
			{
				pMI->BaseAddr = ME.modBaseAddr;
				pMI->Size = ME.modBaseSize;
				CloseHandle(hSnapshot);
				return true;
			}
		} while (Module32Next(hSnapshot, &ME));
	}
	CloseHandle(hSnapshot);
	return false;
}

bool PatchHW(void *BaseAddr, DWORD BaseSize)
{
	DWORD CallAddr, Oldp;
	BYTE *P1, *P2, *Start;
	Start=(BYTE*)FindMemory((char*)BaseAddr, (char*)BaseAddr+BaseSize-1, Flag, 9);
	if(Start)
	{
		P2 = &PatchCodeSpace[0]+0x10;
		do
		{
			P1 = Start - 0x20;
			if  (  P1[0x2F]==0xA1 && P1[0x4C]==0x6A && P1[0x4D]==0x00 && P1[0x4E]==0x56 && P1[0x4F]==0xE8
				&& P1[0x54]==0x83 && P1[0x55]==0xC4 && P1[0x56]==0x08 && P1[0x57]==0x5E && P1[0x58]==0xC3 )
			{
				CallAddr = DWORD(*(DWORD*)(P1+0x50)+DWORD(P1+0x50)+4);

				VirtualProtect(P1+0x4C, 13, PAGE_EXECUTE_READWRITE, &Oldp);

				// JMP P2
				*(P1+0x4C) = 0xE9;
				*(DWORD*)(P1+0x4D) = DWORD(DWORD(P2)-DWORD(P1+0x4D)-4);

				// NOP
				// ..
				// NOP
				memset(P1+0x51, 0x90, 8);

				VirtualProtect(P1+0x4C, 13, Oldp, &Oldp);

				VirtualProtect(&PatchCodeSpace[0], sizeof(PatchCodeSpace), PAGE_EXECUTE_READWRITE, &Oldp);

				// PUSH ESI
				*P2 = 0x56;

				// PUSH DWORD PTR [[P1+0x30]]
				// P1+0x30 is address of the Edict of the player
				*(WORD*)(P2+0x01) = 0x35FF;
				*(DWORD*)(P2+0x03) = *(DWORD*)(P1+0x30);

				 // CALL HookClientCommand
				*(P2+0x07) = 0xE8;
				*(DWORD*)(P2+0x08) = DWORD(DWORD(HookClientEngineCommand)-DWORD(P2+0x08)-4);

				// ADD ESP, 08
				*(WORD*)(P2+0x0C) = 0xC483;
				*(P2+0x0E) = 0x08;

				// TEST EAX, EAX
				*(WORD*)(P2+0x0F) = 0xC085;

				// JNZ +0B
				*(WORD*)(P2+0x11) = 0x0B75;

				// PUSH 0
				*(WORD*)(P2+0x13) = 0x006A;

				// PUSH ESI
				*(P2+0x15) = 0x56;

				// CALL CallAddr
				*(P2+0x16) = 0xE8;
				*(DWORD*)(P2+0x17) = CallAddr-DWORD(P2+0x17)-4;

				// ADD ESP, 8
				*(WORD*)(P2+0x1B) = 0xC483;
				*(P2+0x1D) = 0x08;

				// POP ESI
				*(P2+0x1E) = 0x5E;

				// RETN
				*(P2+0x1F) = 0xC3;

				return true;
			}

		} while ((Start=(BYTE*)FindMemory((char*)Start+9, (char*)BaseAddr+BaseSize-1, Flag, 9)) == NULL);
	}
	return false;
}

bool PatchSW(void *BaseAddr, DWORD BaseSize)
{
	DWORD CallAddr, Oldp;
	BYTE *P1, *P2, *Start;
	Start=(BYTE*)FindMemory((char*)BaseAddr, (char*)BaseAddr+BaseSize-1, Flag, 9);
	if(Start)
	{
		P2 = &PatchCodeSpace[0]+0x10;
		do
		{
			P1 = Start - 0x20;
			if (   P1[0x2F]==0xA1 && P1[0x4A]==0x6A && P1[0x4B]==0x00 && P1[0x4C]==0x56 && P1[0x4D]==0xE8
				&& P1[0x52]==0x83 && P1[0x53]==0xC4 && P1[0x54]==0x08 && P1[0x55]==0x5E && P1[0x56]==0xC3 )
			{
				CallAddr = DWORD(*(DWORD*)(P1+0x4E)+DWORD(P1+0x4E)+4);

				VirtualProtect(P1+0x4A, 13, PAGE_EXECUTE_READWRITE, &Oldp);
				
				// JMP P2
				*(P1+0x4A) = 0xE9;
				*(DWORD*)(P1+0x4B) = DWORD(DWORD(P2)-DWORD(P1+0x4B)-4);

				// NOP
				// ..
				// NOP
				memset(P1+0x4F, 0x90, 8);

				VirtualProtect(P1+0x4A, 13, Oldp, &Oldp);

				VirtualProtect(&PatchCodeSpace[0], sizeof(PatchCodeSpace), PAGE_EXECUTE_READWRITE, &Oldp);

				// PUSH ESI
				*P2 = 0x56;

				// PUSH DWORD PTR [[P1+0x30]]
				// P1+0x30 is address of the Edict of the player
				*(WORD*)(P2+0x01) = 0x35FF;
				*(DWORD*)(P2+0x03) = *(DWORD*)(P1+0x30);

				 // CALL HookClientCommand
				*(P2+0x07) = 0xE8;
				*(DWORD*)(P2+0x08) = DWORD(DWORD(HookClientEngineCommand)-DWORD(P2+0x08)-4);

				// ADD ESP, 08
				*(WORD*)(P2+0x0C) = 0xC483;
				*(P2+0x0E) = 0x08;

				// TEST EAX, EAX
				*(WORD*)(P2+0x0F) = 0xC085;

				// JNZ +0B
				*(WORD*)(P2+0x11) = 0x0B75;

				// PUSH 0
				*(WORD*)(P2+0x13) = 0x006A;

				// PUSH ESI
				*(P2+0x15) = 0x56;

				// CALL CallAddr
				*(P2+0x16) = 0xE8;
				*(DWORD*)(P2+0x17) = CallAddr-DWORD(P2+0x17)-4;

				// ADD ESP, 8
				*(WORD*)(P2+0x1B) = 0xC483;
				*(P2+0x1D) = 0x08;

				// POP ESI
				*(P2+0x1E) = 0x5E;

				// RETN
				*(P2+0x1F) = 0xC3;

				return true;
			}

		} while ((Start=(BYTE*)FindMemory((char*)Start+9, (char*)BaseAddr+BaseSize-1, Flag, 9)) == NULL);
	}
	return false;
}
#endif

#ifdef LINUX

char MBuffer[0x8000];

const char* ExtractFileName(const char *fpath) {
	int sl = strlen(fpath);
	const char *cp = fpath + sl;
	while ((DWORD)cp > (DWORD)fpath) {
		if (*cp == '\\' || *cp == '/') {
			return cp+1;
		}
		cp--;
	}
	return cp;
}

void* LocateLib(const char* libname)
{
	char fname[128], linebuf[512], clib[256];
	const char *clp;
	FILE *fl;
	int sl;
	void *RegStart, *RegEnd;
	Dl_info dli;

	sprintf(fname, "/proc/%d/maps", getpid());
	fl = fopen(fname, "r");
	if (fl == NULL) {
		return NULL;
	}

	setbuffer(fl, MBuffer, sizeof(MBuffer));
	while (fgets(linebuf, sizeof(linebuf), fl))
	{
		sl = sscanf(linebuf, "%x-%x %s %s %s %s %s", &RegStart, &RegEnd, fname, fname, fname, fname, clib);
		if (sl != 7)
			continue;

		if (dladdr(RegStart, &dli) == 0)
			continue;

		clp = ExtractFileName(dli.dli_fname);
		if (!strcmp(libname, clp)) {
			fclose(fl);
			return dli.dli_fbase;
		}
	}
	fclose(fl);
	return NULL;
}

DWORD GetFileSize(const char *FileName)
{
	FILE *fl = fopen(FileName, "rb");
	if (fl == NULL)	return 0;
	int FileSize;
	fseek(fl, 0, SEEK_END);
	FileSize = ftell(fl);
	fseek(fl, 0, SEEK_SET);	
	fclose(fl);
	if (FileSize<0)
		return 0;
	return (DWORD)FileSize;
}

bool GetModuleInfo(const char *ModulePath, CModuleInfo *pMI )
{
	const char *ModuleName = ExtractFileName(ModulePath);
	void *lib = LocateLib(ModuleName);
	if (lib == NULL)
			return false;

	DWORD FileSize = GetFileSize(ModulePath);
	if( FileSize==0 )
		return false;
	pMI->BaseAddr = lib;	
	pMI->Size = FileSize;
	return true;
}

bool MProtect_Ex(void *addr, int npages) {
	void *paddr;
	paddr = (void *)(((size_t)addr) & ~(PAGESIZE-1));
	return !mprotect(paddr, PAGESIZE*(npages+1), PROT_READ | PROT_WRITE | PROT_EXEC);
}

bool PatchEngineAmd(void *BaseAddr, DWORD BaseSize)
{
	DWORD CallAddr;
	BYTE *P1, *P2, *Start;
	Start=(BYTE*)FindMemory((char*)BaseAddr, (char*)BaseAddr+BaseSize-1, Flag, 9);
	if(Start)
	{
		P2 = &PatchCodeSpace[0]+0x10;
		do
		{
			P1 = Start - 0x1C;
			if (   P1[0x00]==0x83 && P1[0x01]==0xC4 && P1[0x02]==0xF8 && P1[0x03]==0x6A && P1[0x04]==0x00
				&& P1[0x05]==0x56 && P1[0x06]==0xE8 && P1[0x0B]==0x83 && P1[0x0C]==0xC4 && P1[0x0D]==0x10
				&& P1[0x0E]==0xEB && P1[0x0F]==0x37 && P1[0x10]==0x83 && P1[0x11]==0xC4 && P1[0x12]==0xF4
				&& P1[0x13]==0x56 && P1[0x14]==0xE8 && P1[0x19]==0x83 && P1[0x1A]==0xC4 && P1[0x1B]==0x10
				&& P1[0x25]==0x83 && P1[0x26]==0xC4 && P1[0x27]==0xF4 && P1[0x28]==0x56 && P1[0x29]==0xE8
				&& P1[0x2E]==0x8B && P1[0x2F]==0x83 && P1[0x34]==0x8B && P1[0x35]==0x93 && P1[0x3A]==0x83
				&& P1[0x3B]==0xC4 && P1[0x3C]==0xF4 && P1[0x3D]==0xFF && P1[0x3E]==0x32 && P1[0x3F]==0x8B
				&& P1[0x40]==0x40 && P1[0x41]==0x4C && P1[0x42]==0xFF && P1[0x43]==0xD0 && P1[0x44]==0x83
				&& P1[0x45]==0xC4 && P1[0x46]==0x20 && P1[0x47]==0x5B && P1[0x48]==0x5E && P1[0x49]==0x5F
				&& P1[0x4A]==0x83 && P1[0x4B]==0xC4 && P1[0x4C]==0x10 && P1[0x4D]==0xC3 )
			{
				CallAddr = DWORD(*(DWORD*)(P1+0x07)+DWORD(P1+0x07)+4);

				MProtect_Ex(P1, 1);
				
				// JMP P2
				*(P1) = 0xE9;
				*(DWORD*)(P1+0x01) = DWORD(DWORD(P2)-DWORD(P1+0x01)-4);

				// NOP
				// ..
				// NOP
				memset(P1+0x05, 0x90, 11);

				MProtect_Ex(&PatchCodeSpace[0], 1);

				// ADD ESP, FFFFFFF8
				*(WORD*)P2 = 0xC483;
				*(P2+0x02) = 0xF8;
				
				// PUSH ESI
				*(P2+0x03) = 0x56;

				// MOV EDX, DWORD PTR [ebx+[P1+0x36]]
				// P1+0x36 POINT TO address of the Edict of the player
				*(WORD*)(P2+0x04) = 0x938B;
				*(DWORD*)(P2+0x06) = *(DWORD*)(P1+0x36);

				// PUSH DWORD PTR [EDX]
				*(WORD*)(P2+0x0A) = 0x32FF;

				 // CALL HookClientCommand
				*(P2+0x0C) = 0xE8;
				*(DWORD*)(P2+0x0D) = DWORD(DWORD(HookClientEngineCommand)-DWORD(P2+0x0D)-4);

				// ADD ESP, 10
				*(WORD*)(P2+0x11) = 0xC483;
				*(P2+0x13) = 0x10;

				// TEST EAX, EAX
				*(WORD*)(P2+0x14) = 0xC085;

				// JNZ +0E
				*(WORD*)(P2+0x16) = 0x0E75;

				// ADD ESP, FFFFFFF8
				*(WORD*)(P2+0x18) = 0xC483;
				*(P2+0x1A) = 0xF8;

				// PUSH 0
				*(WORD*)(P2+0x1B) = 0x006A;

				// PUSH ESI
				*(P2+0x1D) = 0x56;

				// CALL CallAddr
				*(P2+0x1E) = 0xE8;
				*(DWORD*)(P2+0x1F) = DWORD(CallAddr-DWORD(P2+0x1F)-4);

				// ADD ESP, 10
				*(WORD*)(P2+0x23) = 0xC483;
				*(P2+0x25) = 0x10;

				// JMP P1+0x47
				*(P2+0x26) = 0xE9;
				*(DWORD*)(P2+0x27) = DWORD(DWORD(P1+0x47)-DWORD(P2+0x27)-4);

				return true;
			}

		} while ((Start=(BYTE*)FindMemory((char*)Start+9, (char*)BaseAddr+BaseSize-1, Flag, 9)) == NULL);
	}
	return false;
}

bool PatchEngineI486(void *BaseAddr, DWORD BaseSize)
{
	DWORD CallAddr;
	BYTE *P1, *P2, *Start;
	Start=(BYTE*)FindMemory((char*)BaseAddr, (char*)BaseAddr+BaseSize-1, Flag, 9);
	if(Start)
	{
		P2 = &PatchCodeSpace[0]+0x10;
		do
		{
			P1 = Start - 0x1C;
			if (   P1[0x00]==0x83 && P1[0x01]==0xC4 && P1[0x02]==0xF8 && P1[0x03]==0x6A && P1[0x04]==0x00
				&& P1[0x05]==0x56 && P1[0x06]==0xE8 && P1[0x0B]==0xEB && P1[0x0C]==0x38 && P1[0x0D]==0x8D
				&& P1[0x0E]==0x76 && P1[0x0F]==0x00 && P1[0x10]==0x83 && P1[0x11]==0xC4 && P1[0x12]==0xF4
				&& P1[0x13]==0x56 && P1[0x14]==0xE8 && P1[0x19]==0x83 && P1[0x1A]==0xC4 && P1[0x1B]==0x10
				&& P1[0x25]==0x83 && P1[0x26]==0xC4 && P1[0x27]==0xF4 && P1[0x28]==0x56 && P1[0x29]==0xE8
				&& P1[0x2E]==0x83 && P1[0x2F]==0xC4 && P1[0x30]==0xF4 && P1[0x31]==0x8B && P1[0x32]==0x83
				&& P1[0x37]==0x8B && P1[0x38]==0x93 && P1[0x3D]==0x8B && P1[0x3E]==0x00 && P1[0x3F]==0x50
				&& P1[0x40]==0x8B && P1[0x41]==0x42 && P1[0x42]==0x4C && P1[0x43]==0xFF && P1[0x44]==0xD0
				&& P1[0x45]==0x8D && P1[0x46]==0x65 && P1[0x47]==0xE8 && P1[0x48]==0x5B && P1[0x49]==0x5E
				&& P1[0x4A]==0x5F && P1[0x4B]==0x89 && P1[0x4C]==0xEC && P1[0x4D]==0x5D && P1[0x4E]==0xC3 )
			{
				CallAddr = DWORD(*(DWORD*)(P1+0x07)+DWORD(P1+0x07)+4);

				MProtect_Ex(P1, 1);
				
				// JMP P2
				*(P1) = 0xE9;
				*(DWORD*)(P1+0x01) = DWORD(DWORD(P2)-DWORD(P1+0x01)-4);

				// NOP
				// ..
				// NOP
				memset(P1+0x05, 0x90, 8);

				MProtect_Ex(&PatchCodeSpace[0], 1);

				// ADD ESP, FFFFFFF8
				*(WORD*)P2 = 0xC483;
				*(P2+0x02) = 0xF8;
				
				// PUSH ESI
				*(P2+0x03) = 0x56;

				// MOV EAX, DWORD PTR [ebx+[P1+0x33]]
				// P1+0x33 POINT TO address of the Edict of the player
				*(WORD*)(P2+0x04) = 0x838B;
				*(DWORD*)(P2+0x06) = *(DWORD*)(P1+0x33);

				// PUSH DWORD PTR [EAX]
				*(WORD*)(P2+0x0A) = 0x30FF;

				 // CALL HookClientCommand
				*(P2+0x0C) = 0xE8;
				*(DWORD*)(P2+0x0D) = DWORD(DWORD(HookClientEngineCommand)-DWORD(P2+0x0D)-4);

				// ADD ESP, 10
				*(WORD*)(P2+0x11) = 0xC483;
				*(P2+0x13) = 0x10;

				// TEST EAX, EAX
				*(WORD*)(P2+0x14) = 0xC085;

				// JNZ +0B
				*(WORD*)(P2+0x16) = 0x0B75;

				// ADD ESP, FFFFFFF8
				*(WORD*)(P2+0x18) = 0xC483;
				*(P2+0x1A) = 0xF8;

				// PUSH 0
				*(WORD*)(P2+0x1B) = 0x006A;

				// PUSH ESI
				*(P2+0x1D) = 0x56;

				// CALL CallAddr
				*(P2+0x1E) = 0xE8;
				*(DWORD*)(P2+0x1F) = DWORD(CallAddr-DWORD(P2+0x1F)-4);

				// JMP P1+0x45
				*(P2+0x26) = 0xE9;
				*(DWORD*)(P2+0x27) = DWORD(DWORD(P1+0x45)-DWORD(P2+0x27)-4);

				return true;
			}

		} while ((Start=(BYTE*)FindMemory((char*)Start+9, (char*)BaseAddr+BaseSize-1, Flag, 9)) == NULL);
	}
	return false;
}

bool PatchEngineI686(void *BaseAddr, DWORD BaseSize)
{
	DWORD CallAddr;
	BYTE *P1, *P2, *Start;
	Start=(BYTE*)FindMemory((char*)BaseAddr, (char*)BaseAddr+BaseSize-1, Flag, 9);
	if(Start)
	{
		P2 = &PatchCodeSpace[0]+0x10;
		do
		{
			P1 = Start - 0x1C;
			if (   P1[0x00]==0x83 && P1[0x01]==0xC4 && P1[0x02]==0xF8 && P1[0x03]==0x6A && P1[0x04]==0x00
				&& P1[0x05]==0x56 && P1[0x06]==0xE8 && P1[0x0B]==0x83 && P1[0x0C]==0xC4 && P1[0x0D]==0x10
				&& P1[0x0E]==0xEB && P1[0x0F]==0x38 && P1[0x10]==0x83 && P1[0x11]==0xC4 && P1[0x12]==0xF4
				&& P1[0x13]==0x56 && P1[0x14]==0xE8 && P1[0x19]==0x83 && P1[0x1A]==0xC4 && P1[0x1B]==0x10
				&& P1[0x25]==0x83 && P1[0x26]==0xC4 && P1[0x27]==0xF4 && P1[0x28]==0x56 && P1[0x29]==0xE8
				&& P1[0x2E]==0x8B && P1[0x2F]==0x83 && P1[0x34]==0x8B && P1[0x35]==0x93 && P1[0x3A]==0x83
				&& P1[0x3B]==0xC4 && P1[0x3C]==0xF4 && P1[0x3D]==0x8B && P1[0x3E]==0x00 && P1[0x3F]==0x50
				&& P1[0x40]==0x8B && P1[0x41]==0x42 && P1[0x42]==0x4C && P1[0x43]==0xFF && P1[0x44]==0xD0
				&& P1[0x45]==0x83 && P1[0x46]==0xC4 && P1[0x47]==0x20 && P1[0x48]==0x5B && P1[0x49]==0x5E
				&& P1[0x4A]==0x5F && P1[0x4B]==0x83 && P1[0x4C]==0xC4 && P1[0x4D]==0x10 && P1[0x4E]==0xC3 )
			{
				CallAddr = DWORD(*(DWORD*)(P1+0x07)+DWORD(P1+0x07)+4);

				MProtect_Ex(P1, 1);
				
				// JMP P2
				*(P1) = 0xE9;
				*(DWORD*)(P1+0x01) = DWORD(DWORD(P2)-DWORD(P1+0x01)-4);

				// NOP
				// ..
				// NOP
				memset(P1+0x05, 0x90, 11);

				MProtect_Ex(&PatchCodeSpace[0], 1);

				// ADD ESP, FFFFFFF8
				*(WORD*)P2 = 0xC483;
				*(P2+0x02) = 0xF8;
				
				// PUSH ESI
				*(P2+0x03) = 0x56;

				// MOV EAX, DWORD PTR [ebx+[P1+0x30]]
				// P1+0x30 POINT TO address of the Edict of the player
				*(WORD*)(P2+0x04) = 0x838B;
				*(DWORD*)(P2+0x06) = *(DWORD*)(P1+0x30);

				// PUSH DWORD PTR [EAX]
				*(WORD*)(P2+0x0A) = 0x30FF;

				 // CALL HookClientCommand
				*(P2+0x0C) = 0xE8;
				*(DWORD*)(P2+0x0D) = DWORD(DWORD(HookClientEngineCommand)-DWORD(P2+0x0D)-4);

				// ADD ESP, 10
				*(WORD*)(P2+0x11) = 0xC483;
				*(P2+0x13) = 0x10;

				// TEST EAX, EAX
				*(WORD*)(P2+0x14) = 0xC085;

				// JNZ +0E
				*(WORD*)(P2+0x16) = 0x0E75;

				// ADD ESP, FFFFFFF8
				*(WORD*)(P2+0x18) = 0xC483;
				*(P2+0x1A) = 0xF8;

				// PUSH 0
				*(WORD*)(P2+0x1B) = 0x006A;

				// PUSH ESI
				*(P2+0x1D) = 0x56;

				// CALL CallAddr
				*(P2+0x1E) = 0xE8;
				*(DWORD*)(P2+0x1F) = DWORD(CallAddr-DWORD(P2+0x1F)-4);

				// ADD ESP, 10
				*(WORD*)(P2+0x23) = 0xC483;
				*(P2+0x25) = 0x10;

				// JMP P1+0x48
				*(P2+0x26) = 0xE9;
				*(DWORD*)(P2+0x27) = DWORD(DWORD(P1+0x48)-DWORD(P2+0x27)-4);

				return true;
			}

		} while ((Start=(BYTE*)FindMemory((char*)Start+9, (char*)BaseAddr+BaseSize-1, Flag, 9)) == NULL);
	}
	return false;
}
#endif

C_DLLEXPORT int Meta_Attach(PLUG_LOADTIME  now, META_FUNCTIONS *pFunctionTable,
					meta_globals_t *pMGlobals, gamedll_funcs_t *pGamedllFuncs) 
{
	if(now>MODULE_LOADABLE)
	{
		LOG_ERROR(PLID, "Can't load plugin right now");
		return FALSE;
	}
	if(!pMGlobals) {
		LOG_ERROR(PLID, "Meta_Attach called with null pMGlobals");
		return FALSE;
	}
	gpMetaGlobals=pMGlobals;
	if(!pFunctionTable) {
		LOG_ERROR(PLID, "Meta_Attach called with null pFunctionTable");
		return FALSE;
	}
	memcpy(pFunctionTable, &gMetaFunctionTable, sizeof(META_FUNCTIONS));
	gpGamedllFuncs=pGamedllFuncs;

	CVAR_REGISTER(&MODULE_CVAR_VERSION);
	CVAR_REGISTER(&MODULE_CVAR_ENABLE);

	memset(gPlayer, 0, sizeof(gPlayer));
	memset(PatchCodeSpace, 0x90, sizeof(PatchCodeSpace));

	bool bSuccess = false;
	CModuleInfo MI;

#ifdef _WIN32
	if(IS_DEDICATED_SERVER())
	{
		if(GetModuleInfo("swds.dll", &MI))
			bSuccess = PatchSW(MI.BaseAddr, MI.Size);
	} else {
		if(GetModuleInfo(NULL, &MI))
			bSuccess = PatchHW(MI.BaseAddr, MI.Size) || PatchSW((char*)MI.BaseAddr, MI.Size);
	}
#endif

#ifdef LINUX
	if (GetModuleInfo("./engine_amd.so", &MI))
		bSuccess = PatchEngineAmd(MI.BaseAddr, MI.Size);
	else if (GetModuleInfo("./engine_i486.so",&MI))
		bSuccess = PatchEngineI486(MI.BaseAddr, MI.Size);
	else if (GetModuleInfo("./engine_i686.so",&MI))
		bSuccess = PatchEngineI686(MI.BaseAddr, MI.Size);
#endif

	SERVER_PRINT("\n");
	if( bSuccess )
		SERVER_PRINT("[ANTI-DLFILE] Patch \"cmd dlfile\" successful!\n");
	else
		SERVER_PRINT("[ANTI-DLFILE] Patch \"cmd dlfile\" failed.\n");
	SERVER_PRINT("\n");
	return TRUE;
}

C_DLLEXPORT int Meta_Detach(PLUG_LOADTIME now, PL_UNLOAD_REASON reason) 
{
	if (now>MODULE_UNLOADABLE && reason!=PNL_CMD_FORCED)
	{
		LOG_ERROR(PLID, "Can't unload plugin right now");
		return FALSE;
	}
	return TRUE;
}
